"use client";
import Link from "next/link";
import { useEffect, useState } from "react";
import { useSession } from "next-auth/react";
import { useRouter } from "next/router";
import { useLanguage } from "@/contexts/LanguageContext";
import {
  RefreshCw,
  AlertTriangle,
  CheckCircle2,
  Clock,
  Calendar,
  Building2,
  MapPin,
  Wrench,
  Search,
  Filter,
  X,
  ChevronDown,
  ChevronUp,
  Copy,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from "@/components/ui/tooltip";

type Task = {
  id: string;
  title: string;
  description: string;
  dueDate: string | null;
  completedAt: string | null;
  currentPeriodCompletedAt?: string | null; // For recurring tasks: current period completion
  isRecurring: boolean;
  recurrencePattern: string | null;
  createdAt: string;
  assignedTo: {
    id: string;
    name: string | null;
    email: string;
  } | null;
  department: {
    id: string;
    name: string;
  } | null;
  location: {
    id: string;
    name: string;
  } | null;
  element: {
    id: string;
    name: string;
  } | null;
};

type Department = { id: string; name: string };
type Location = { id: string; name: string; departmentId: string };
type Element = { id: string; name: string; locationId: string };
type Technician = { 
  id: string; 
  name: string | null; 
  email: string;
  isExternal?: boolean;
  companyName?: string | null;
};

export default function TasksList() {
  const { data: session, status } = useSession();
  const router = useRouter();
  const { t } = useLanguage();
  const [tasks, setTasks] = useState<Task[]>([]);
  const [loading, setLoading] = useState(true);
  const [err, setErr] = useState<string | null>(null);
  const [filter, setFilter] = useState<"all" | "pending" | "completed">("all");
  const [showFilters, setShowFilters] = useState(false);
  const [completingId, setCompletingId] = useState<string | null>(null);
  const [duplicatingId, setDuplicatingId] = useState<string | null>(null);

  // Filter states
  const [search, setSearch] = useState("");
  const [departmentId, setDepartmentId] = useState<string>("");
  const [locationId, setLocationId] = useState<string>("");
  const [elementId, setElementId] = useState<string>("");
  const [assignedToId, setAssignedToId] = useState<string>("");
  const [dateFrom, setDateFrom] = useState<string>("");
  const [dateTo, setDateTo] = useState<string>("");
  const [hasNotOk, setHasNotOk] = useState<boolean | null>(null);

  // Data for filters
  const [departments, setDepartments] = useState<Department[]>([]);
  const [locations, setLocations] = useState<Location[]>([]);
  const [elements, setElements] = useState<Element[]>([]);
  const [technicians, setTechnicians] = useState<Technician[]>([]);

  // Track the last URL query to detect navigation changes
  const [lastQueryString, setLastQueryString] = useState<string>("");

  // Initialize filters from URL query parameters
  useEffect(() => {
    if (!router.isReady) return;

    const query = router.query;
    const currentQueryString = new URLSearchParams(
      Object.entries(query).reduce((acc, [key, value]) => {
        if (value && typeof value === "string") {
          acc[key] = value;
        }
        return acc;
      }, {} as Record<string, string>)
    ).toString();

    // Only update filters if the URL query actually changed (navigation back/forward)
    if (currentQueryString !== lastQueryString) {
      if (query.status)
        setFilter(query.status as "all" | "pending" | "completed");
      else setFilter("all");

      setSearch((query.search as string) || "");
      setDepartmentId((query.departmentId as string) || "");
      setLocationId((query.locationId as string) || "");
      setElementId((query.elementId as string) || "");
      setAssignedToId((query.assignedToId as string) || "");
      setDateFrom((query.dateFrom as string) || "");
      setDateTo((query.dateTo as string) || "");
      if (query.hasNotOk) {
        setHasNotOk(
          query.hasNotOk === "true"
            ? true
            : query.hasNotOk === "false"
            ? false
            : null
        );
      } else {
        setHasNotOk(null);
      }

      setLastQueryString(currentQueryString);
    }
  }, [router.isReady, router.query, lastQueryString]);

  // Update URL when filters change (but not when URL is the source of truth)
  useEffect(() => {
    if (!router.isReady) return;

    const params = new URLSearchParams();
    if (filter !== "all") {
      params.append("status", filter);
    }
    if (search) {
      params.append("search", search);
    }
    if (departmentId) {
      params.append("departmentId", departmentId);
    }
    if (locationId) {
      params.append("locationId", locationId);
    }
    if (elementId) {
      params.append("elementId", elementId);
    }
    if (assignedToId) {
      params.append("assignedToId", assignedToId);
    }
    if (dateFrom) {
      params.append("dateFrom", dateFrom);
    }
    if (dateTo) {
      params.append("dateTo", dateTo);
    }
    if (hasNotOk !== null) {
      params.append("hasNotOk", hasNotOk ? "true" : "false");
    }

    const queryString = params.toString();
    const newUrl = queryString ? `/tasks?${queryString}` : "/tasks";
    const currentPath = router.asPath.split("?")[0];
    const currentQuery = router.asPath.includes("?")
      ? router.asPath.split("?")[1]
      : "";

    // Only update URL if it's different to avoid infinite loops
    // Also update lastQueryString to prevent re-initialization
    if (currentPath !== "/tasks" || currentQuery !== queryString) {
      setLastQueryString(queryString);
      router.replace(newUrl, undefined, { shallow: true });
    }
  }, [
    filter,
    search,
    departmentId,
    locationId,
    elementId,
    assignedToId,
    dateFrom,
    dateTo,
    hasNotOk,
    router.isReady,
  ]);

  // Load departments
  useEffect(() => {
    if (status !== "authenticated") return;
    fetch("/api/departments")
      .then((r) => r.json())
      .then((j) => {
        if (j.ok) {
          setDepartments(j.items || []);
        }
      })
      .catch((e) => console.error("Failed to load departments:", e));
  }, [status]);

  // Load technicians (admin only)
  useEffect(() => {
    if (status !== "authenticated") return;
    const role = (session?.user as any)?.role?.toUpperCase() || "";
    if (role === "ADMIN") {
      fetch("/api/admin/users?role=technician")
        .then((r) => r.json())
        .then((j) => {
          if (j.ok) {
            setTechnicians(j.users || []);
          }
        })
        .catch((e) => console.error("Failed to load technicians:", e));
    }
  }, [status, session]);

  // Load locations when department changes
  useEffect(() => {
    if (!departmentId) {
      setLocations([]);
      if (!router.query.locationId) {
        setLocationId("");
      }
      setElements([]);
      if (!router.query.elementId) {
        setElementId("");
      }
      return;
    }
    fetch(`/api/locations?departmentId=${departmentId}`)
      .then((r) => r.json())
      .then((j) => {
        if (j.ok) {
          setLocations(j.items || []);
          // Only clear locationId if it's not in URL and not valid for this department
          if (!router.query.locationId) {
            const currentLocation = locations.find((l) => l.id === locationId);
            if (
              !currentLocation ||
              currentLocation.departmentId !== departmentId
            ) {
              setLocationId("");
            }
          }
        }
      })
      .catch((e) => console.error("Failed to load locations:", e));
  }, [departmentId]);

  // Load elements when location changes
  useEffect(() => {
    if (!locationId) {
      setElements([]);
      if (!router.query.elementId) {
        setElementId("");
      }
      return;
    }
    fetch(`/api/elements?locationId=${locationId}`)
      .then((r) => r.json())
      .then((j) => {
        if (j.ok) {
          setElements(j.items || []);
          // Only clear elementId if it's not in URL and not valid for this location
          if (!router.query.elementId) {
            const currentElement = elements.find((e) => e.id === elementId);
            if (!currentElement || currentElement.locationId !== locationId) {
              setElementId("");
            }
          }
        }
      })
      .catch((e) => console.error("Failed to load elements:", e));
  }, [locationId]);

  const fetchTasks = () => {
    if (status !== "authenticated") return;
    setLoading(true);
    const params = new URLSearchParams();
    // Don't filter by status in API - we'll filter in frontend to get accurate counts
    // if (filter !== "all") {
    //   params.append("status", filter);
    // }
    if (search) {
      params.append("search", search);
    }
    if (departmentId) {
      params.append("departmentId", departmentId);
    }
    if (locationId) {
      params.append("locationId", locationId);
    }
    if (elementId) {
      params.append("elementId", elementId);
    }
    if (assignedToId) {
      params.append("assignedToId", assignedToId);
    }
    if (dateFrom) {
      params.append("dateFrom", dateFrom);
    }
    if (dateTo) {
      params.append("dateTo", dateTo);
    }
    if (hasNotOk !== null) {
      params.append("hasNotOk", hasNotOk ? "true" : "false");
    }
    const query = params.toString();
    const url = query ? `/api/tasks?${query}` : "/api/tasks";
    fetch(url)
      .then((r) => r.json())
      .then((j) => {
        if (!j.ok) throw new Error(j.error || "Failed to load tasks");
        setTasks(j.tasks || []);
      })
      .catch((e) => setErr(String(e)))
      .finally(() => setLoading(false));
  };

  useEffect(() => {
    if (!router.isReady) return;
    fetchTasks();
  }, [
    status,
    router.isReady,
    filter,
    search,
    departmentId,
    locationId,
    elementId,
    assignedToId,
    dateFrom,
    dateTo,
    hasNotOk,
  ]);

  // Auto-refresh every 30 seconds
  useEffect(() => {
    if (status !== "authenticated" || !router.isReady) return;
    const interval = setInterval(() => {
      fetchTasks();
    }, 30000);
    return () => clearInterval(interval);
  }, [
    status,
    router.isReady,
    filter,
    search,
    departmentId,
    locationId,
    elementId,
    assignedToId,
    dateFrom,
    dateTo,
    hasNotOk,
  ]);

  const handleComplete = async (taskId: string, completed: boolean) => {
    setCompletingId(taskId);
    try {
      const res = await fetch(`/api/tasks/${taskId}`, {
        method: "PATCH",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ completed }),
      });
      const json = await res.json();
      if (json.ok) {
        fetchTasks();
      } else {
        setErr(json.error || "Failed to update task");
      }
    } catch (error: any) {
      setErr(error?.message || "Failed to update task");
    } finally {
      setCompletingId(null);
    }
  };

  const handleDuplicate = async (taskId: string) => {
    setDuplicatingId(taskId);
    setErr(null);
    try {
      const res = await fetch(`/api/tasks/${taskId}/duplicate`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
      });
      const json = await res.json();
      if (json.ok) {
        // Refresh tasks to show the new duplicate
        fetchTasks();
        // Optionally navigate to the new task
        // router.push(`/tasks/${json.task.id}`);
      } else {
        setErr(json.error || "Failed to duplicate task");
      }
    } catch (error: any) {
      setErr(error?.message || "Failed to duplicate task");
    } finally {
      setDuplicatingId(null);
    }
  };

  const clearFilters = () => {
    setSearch("");
    setDepartmentId("");
    setLocationId("");
    setElementId("");
    setAssignedToId("");
    setDateFrom("");
    setDateTo("");
    setHasNotOk(null);
  };

  const hasActiveFilters =
    search ||
    departmentId ||
    locationId ||
    elementId ||
    assignedToId ||
    dateFrom ||
    dateTo ||
    hasNotOk !== null;

  if (status === "unauthenticated") {
    return (
      <main className='flex flex-col items-center justify-center min-h-[60vh] text-center'>
        <AlertTriangle className='w-10 h-10 mb-3 text-orange-500' />
        <p className='text-lg font-medium mb-4'>
          {t("tasks.mustLogin") || "You need to sign in to view tasks."}
        </p>
        <Link href='/login'>
          <Button>{t("tasks.goToLogin") || "Go to Login"}</Button>
        </Link>
      </main>
    );
  }

  const role = (session?.user as any)?.role?.toUpperCase() || "";
  const isTechnician = role === "TECHNICIAN";
  const isAdmin = role === "ADMIN";

  if (!isTechnician && !isAdmin) {
    return (
      <main className='flex flex-col items-center justify-center min-h-[60vh] text-center'>
        <AlertTriangle className='w-10 h-10 mb-3 text-orange-500' />
        <p className='text-lg font-medium mb-4'>
          {t("tasks.noPermission") || "You don't have permission to view tasks."}
        </p>
      </main>
    );
  }

  // Calculate counts from all tasks (not filtered by status)
  const allTasks = tasks;
  const pendingTasks = allTasks.filter((t) => !t.completedAt);
  const completedTasks = allTasks.filter((t) => t.completedAt);
  
  // Filter displayed tasks based on selected tab
  const displayedTasks = filter === "pending" 
    ? pendingTasks 
    : filter === "completed" 
    ? completedTasks 
    : allTasks;

  return (
    <main className='w-full p-3 sm:p-4 md:p-6 space-y-4 md:space-y-6'>
      {/* Header */}
      <div className='flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3'>
        <div>
          <h1 className='text-2xl sm:text-3xl font-bold'>{t("tasks.title") || "Tasks"}</h1>
          <p className='text-sm sm:text-base text-muted-foreground'>
            {isTechnician
              ? t("tasks.subtitleTechnician") || "View and complete your assigned tasks"
              : t("tasks.subtitleAdmin") || "Manage and assign tasks to technicians"}
          </p>
        </div>

        <div className='flex items-center gap-2 sm:gap-3 flex-wrap'>
          <Button variant='outline' onClick={fetchTasks} disabled={loading} size='sm' className='sm:size-default'>
            <RefreshCw className={`w-4 h-4 ${loading ? "animate-spin" : ""}`} />
            <span className='hidden sm:inline'>{t("tasks.refresh") || "Refresh"}</span>
          </Button>
          {isAdmin && (
            <Link href='/tasks/new'>
              <Button size='sm' className='sm:size-default'>{t("tasks.createTask") || "Create Task"}</Button>
            </Link>
          )}
        </div>
      </div>

      {/* Search Bar */}
      <div className='flex flex-col sm:flex-row items-stretch sm:items-center gap-2'>
        <div className='relative flex-1'>
          <Search className='absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground' />
          <Input
            type='text'
            placeholder={t("tasks.searchPlaceholder") || "Search tasks by title or description..."}
            value={search}
            onChange={(e) => setSearch(e.target.value)}
            className='pl-10 text-sm sm:text-base'
          />
        </div>
        <div className='flex items-center gap-2'>
          <Button
            variant='outline'
            onClick={() => setShowFilters(!showFilters)}
            className='flex items-center gap-2 flex-1 sm:flex-initial'
            size='sm'
          >
            <Filter className='w-4 h-4' />
            <span className='hidden sm:inline'>{t("tasks.filters") || "Filters"}</span>
            {hasActiveFilters && (
              <span className='bg-primary text-primary-foreground rounded-full w-5 h-5 flex items-center justify-center text-xs'>
                {
                  [
                    departmentId,
                    locationId,
                    elementId,
                    assignedToId,
                    dateFrom,
                    dateTo,
                    hasNotOk !== null,
                  ].filter(Boolean).length
                }
              </span>
            )}
            {showFilters ? (
              <ChevronUp className='w-4 h-4' />
            ) : (
              <ChevronDown className='w-4 h-4' />
            )}
          </Button>
          {hasActiveFilters && (
            <Button variant='ghost' size='sm' onClick={clearFilters} className='flex-shrink-0'>
              <X className='w-4 h-4 sm:mr-1' />
              <span className='hidden sm:inline'>{t("tasks.clearFilters") || "Clear Filters"}</span>
            </Button>
          )}
        </div>
      </div>

      {/* Filter Panel */}
      {showFilters && (
        <div className='border rounded-lg p-3 sm:p-4 bg-card space-y-4'>
          <div className='grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3 sm:gap-4'>
            {/* Department Filter */}
            <div className='space-y-2'>
              <label className='text-sm font-medium'>{t("tasks.department") || "Department"}</label>
              <Select
                value={departmentId || "all"}
                onValueChange={(val) =>
                  setDepartmentId(val === "all" ? "" : val)
                }
              >
                <SelectTrigger>
                  <SelectValue placeholder={t("tasks.allDepartments") || "All departments"} />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value='all'>{t("tasks.allDepartments") || "All departments"}</SelectItem>
                  {departments.map((dep) => (
                    <SelectItem key={dep.id} value={dep.id}>
                      {dep.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {/* Location Filter */}
            <div className='space-y-2'>
              <label className='text-sm font-medium'>{t("tasks.location") || "Location"}</label>
              <Select
                value={locationId || "all"}
                onValueChange={(val) => setLocationId(val === "all" ? "" : val)}
                disabled={!departmentId}
              >
                <SelectTrigger>
                  <SelectValue
                    placeholder={
                      !departmentId
                        ? t("tasks.selectDepartmentFirst") || "Select department first"
                        : t("tasks.allLocations") || "All locations"
                    }
                  />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value='all'>{t("tasks.allLocations") || "All locations"}</SelectItem>
                  {locations.map((loc) => (
                    <SelectItem key={loc.id} value={loc.id}>
                      {loc.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {/* Element Filter */}
            <div className='space-y-2'>
              <label className='text-sm font-medium'>{t("tasks.element") || "Element"}</label>
              <Select
                value={elementId || "all"}
                onValueChange={(val) => setElementId(val === "all" ? "" : val)}
                disabled={!locationId}
              >
                <SelectTrigger>
                  <SelectValue
                    placeholder={
                      !locationId ? t("tasks.selectLocationFirst") || "Select location first" : t("tasks.allElements") || "All elements"
                    }
                  />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value='all'>{t("tasks.allElements") || "All elements"}</SelectItem>
                  {elements.map((el) => (
                    <SelectItem key={el.id} value={el.id}>
                      {el.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {/* Assigned To Filter (Admin only) */}
            {isAdmin && (
              <div className='space-y-2'>
                <label className='text-sm font-medium'>{t("tasks.assignedToFilter") || "Assigned To"}</label>
                <Select
                  value={assignedToId || "all"}
                  onValueChange={(val) =>
                    setAssignedToId(val === "all" ? "" : val)
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder={t("tasks.allTechnicians") || "All technicians"} />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value='all'>{t("tasks.allTechnicians") || "All technicians"}</SelectItem>
                    <SelectItem value='unassigned'>{t("tasks.unassigned") || "Unassigned"}</SelectItem>
                    {technicians.map((tech) => {
                      const displayName = tech.name || tech.email;
                      const externalTag = tech.isExternal ? " [External]" : "";
                      const companyInfo = tech.companyName ? ` (${tech.companyName})` : "";
                      return (
                        <SelectItem key={tech.id} value={tech.id}>
                          {displayName}{externalTag}{companyInfo}
                        </SelectItem>
                      );
                    })}
                  </SelectContent>
                </Select>
              </div>
            )}

            {/* Date From Filter */}
            <div className='space-y-2'>
              <label className='text-sm font-medium'>{t("tasks.dateFrom") || "Date From"}</label>
              <Input
                type='date'
                value={dateFrom}
                onChange={(e) => setDateFrom(e.target.value)}
              />
            </div>

            {/* Date To Filter */}
            <div className='space-y-2'>
              <label className='text-sm font-medium'>{t("tasks.dateTo") || "Date To"}</label>
              <Input
                type='date'
                value={dateTo}
                onChange={(e) => setDateTo(e.target.value)}
              />
            </div>

            {/* Has NOT OK Filter */}
            <div className='space-y-2'>
              <label className='text-sm font-medium'>{t("tasks.hasNotOkItems") || "Has NOT OK Items"}</label>
              <Select
                value={
                  hasNotOk === null
                    ? "all"
                    : hasNotOk === true
                    ? "true"
                    : "false"
                }
                onValueChange={(val) =>
                  setHasNotOk(
                    val === "all" ? null : val === "true" ? true : false
                  )
                }
              >
                <SelectTrigger>
                  <SelectValue placeholder={t("tasks.allTasks") || "All tasks"} />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value='all'>{t("tasks.allTasks") || "All tasks"}</SelectItem>
                  <SelectItem value='true'>{t("tasks.hasNotOkItemsOption") || "Has NOT OK items"}</SelectItem>
                  <SelectItem value='false'>{t("tasks.noNotOkItemsOption") || "No NOT OK items"}</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        </div>
      )}

      {/* Filter Tabs */}
      <div className='flex gap-2 border-b overflow-x-auto'>
        <button
          onClick={() => setFilter("all")}
          className={`px-3 sm:px-4 py-2 font-medium border-b-2 transition whitespace-nowrap text-sm sm:text-base ${
            filter === "all"
              ? "border-primary text-primary"
              : "border-transparent text-muted-foreground hover:text-foreground"
          }`}
        >
          {t("tasks.all") || "All"} ({allTasks.length})
        </button>
        <button
          onClick={() => setFilter("pending")}
          className={`px-3 sm:px-4 py-2 font-medium border-b-2 transition whitespace-nowrap text-sm sm:text-base ${
            filter === "pending"
              ? "border-primary text-primary"
              : "border-transparent text-muted-foreground hover:text-foreground"
          }`}
        >
          {t("tasks.pending") || "Pending"} ({pendingTasks.length})
        </button>
        <button
          onClick={() => setFilter("completed")}
          className={`px-3 sm:px-4 py-2 font-medium border-b-2 transition whitespace-nowrap text-sm sm:text-base ${
            filter === "completed"
              ? "border-primary text-primary"
              : "border-transparent text-muted-foreground hover:text-foreground"
          }`}
        >
          {t("tasks.completed") || "Completed"} ({completedTasks.length})
        </button>
      </div>

      {/* Error */}
      {err && (
        <div className='rounded-lg border-l-4 border-red-500 bg-red-50 p-4 text-red-700 flex items-center gap-2'>
          <AlertTriangle className='w-4 h-4' /> {err}
        </div>
      )}

      {/* Loading */}
      {loading && (
        <div className='flex items-center gap-2 text-muted-foreground'>
          <RefreshCw className='w-4 h-4 animate-spin' />
          {t("tasks.loading") || "Loading tasks..."}
        </div>
      )}

      {/* Tasks List */}
      {!loading && !err && displayedTasks.length > 0 && (
        <div className='space-y-4'>
          {displayedTasks.map((task) => (
            <div
              key={task.id}
              className={`border rounded-lg p-3 sm:p-4 md:p-6 bg-card hover:shadow-md transition ${
                task.completedAt ? "opacity-75" : ""
              }`}
            >
              <div className='flex flex-col sm:flex-row sm:items-start sm:justify-between gap-3 sm:gap-4'>
                <div className='flex-1 min-w-0'>
                  <div className='flex flex-wrap items-center gap-2 mb-2'>
                    <h3
                      className={`text-base sm:text-lg font-semibold break-words ${
                        task.completedAt
                          ? "line-through text-muted-foreground"
                          : ""
                      }`}
                    >
                      {task.title}
                    </h3>
                    <div className='flex items-center gap-2 flex-wrap'>
                      {task.completedAt ? (
                        <span className='px-2 py-1 text-xs bg-green-100 text-green-800 rounded-full flex items-center gap-1 whitespace-nowrap'>
                          <CheckCircle2 className='w-3 h-3' />
                          {t("tasks.completed") || "Completed"}
                        </span>
                      ) : (
                        <span className='px-2 py-1 text-xs bg-yellow-100 text-yellow-800 rounded-full flex items-center gap-1 whitespace-nowrap'>
                          <Clock className='w-3 h-3' />
                          {t("tasks.pending") || "Pending"}
                        </span>
                      )}
                      {task.isRecurring && (
                        <span className='px-2 py-1 text-xs bg-blue-100 text-blue-800 rounded-full whitespace-nowrap'>
                          {t("tasks.recurring") || "Recurring"}
                        </span>
                      )}
                    </div>
                  </div>

                  <p className='text-sm sm:text-base text-muted-foreground mb-3 sm:mb-4 break-words'>
                    {task.description}
                  </p>

                  <div className='flex flex-wrap gap-2 sm:gap-4 text-xs sm:text-sm text-muted-foreground'>
                    {task.assignedTo && (
                      <div className='flex items-center gap-1'>
                        <span className='font-medium'>{t("tasks.assignedTo") || "Assigned to:"}</span>
                        <span className='truncate max-w-[150px] sm:max-w-none'>{task.assignedTo.name || task.assignedTo.email}</span>
                      </div>
                    )}
                    {task.department && (
                      <div className='flex items-center gap-1'>
                        <Building2 className='w-3 h-3 sm:w-4 sm:h-4 flex-shrink-0' />
                        <span className='truncate'>{task.department.name}</span>
                      </div>
                    )}
                    {task.location && (
                      <div className='flex items-center gap-1'>
                        <MapPin className='w-3 h-3 sm:w-4 sm:h-4 flex-shrink-0' />
                        <span className='truncate'>{task.location.name}</span>
                      </div>
                    )}
                    {task.element && (
                      <div className='flex items-center gap-1'>
                        <Wrench className='w-3 h-3 sm:w-4 sm:h-4 flex-shrink-0' />
                        <span className='truncate'>{task.element.name}</span>
                      </div>
                    )}
                    {task.dueDate && (
                      <div className='flex items-center gap-1'>
                        <Calendar className='w-3 h-3 sm:w-4 sm:h-4 flex-shrink-0' />
                        <span>{t("tasks.due") || "Due:"} {new Date(task.dueDate).toLocaleDateString()}</span>
                      </div>
                    )}
                  </div>
                </div>

                <div className='flex items-center gap-2 flex-shrink-0'>
                  <Link href={`/tasks/${task.id}`} className='flex-1 sm:flex-initial'>
                    <Button variant='outline' size='sm' className='w-full sm:w-auto'>
                      {t("tasks.view") || "View"}
                    </Button>
                  </Link>
                  {isAdmin && (
                    <TooltipProvider>
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <Button
                            onClick={() => handleDuplicate(task.id)}
                            disabled={
                              duplicatingId === task.id ||
                              completingId === task.id
                            }
                            size='sm'
                            variant='outline'
                          >
                            {duplicatingId === task.id ? (
                              <RefreshCw className='w-4 h-4 animate-spin' />
                            ) : (
                              <Copy className='w-4 h-4' />
                            )}
                          </Button>
                        </TooltipTrigger>
                        <TooltipContent>
                          <p>
                            {t("tasks.duplicateTooltip") || "Duplicate this task (will create a copy without assigned user)"}
                          </p>
                        </TooltipContent>
                      </Tooltip>
                    </TooltipProvider>
                  )}
                  {!task.completedAt && (
                    <Button
                      onClick={() => handleComplete(task.id, true)}
                      disabled={
                        completingId === task.id || duplicatingId === task.id
                      }
                      size='sm'
                    >
                      {completingId === task.id ? (
                        <RefreshCw className='w-4 h-4 animate-spin' />
                      ) : (
                        <CheckCircle2 className='w-4 h-4' />
                      )}
                      {t("tasks.markComplete") || "Mark Complete"}
                    </Button>
                  )}
                  {task.completedAt && (
                    <Button
                      onClick={() => handleComplete(task.id, false)}
                      disabled={
                        completingId === task.id || duplicatingId === task.id
                      }
                      size='sm'
                      variant='outline'
                    >
                      {completingId === task.id ? (
                        <RefreshCw className='w-4 h-4 animate-spin' />
                      ) : (
                        t("tasks.reopen") || "Reopen"
                      )}
                    </Button>
                  )}
                </div>
              </div>
            </div>
          ))}
        </div>
      )}

      {/* Empty State */}
      {!loading && !err && displayedTasks.length === 0 && (
        <div className='text-center py-12 border border-dashed rounded-lg'>
          <AlertTriangle className='w-12 h-12 mx-auto mb-4 text-muted-foreground' />
          <p className='text-lg font-medium mb-2'>{t("tasks.noTasks") || "No tasks found"}</p>
          <p className='text-muted-foreground'>
            {hasActiveFilters
              ? t("tasks.noTasksFiltered") || "Try adjusting your filters"
              : filter === "pending"
              ? t("tasks.noPendingTasks") || "You have no pending tasks"
              : filter === "completed"
              ? t("tasks.noCompletedTasks") || "You have no completed tasks"
              : t("tasks.noAssignedTasks") || "You have no tasks assigned"}
          </p>
        </div>
      )}
    </main>
  );
}
